#!/bin/bash
# ============================================================================
# Script: CrearInstaladorMac.sh
# Descripción: Crea un instalador completo para macOS con JRE 8 embebido
# Proyecto: Factux Costa Rica - ElectricPos
# Autor: Space Software Costa Rica
# Fecha: Octubre 2025
# ============================================================================

set -e  # Salir si hay errores

# ============================================================================
# CONFIGURACIÓN - AJUSTA ESTOS VALORES SEGÚN TU PROYECTO
# ============================================================================

APP_NAME="Factux Costa Rica"
APP_VERSION="4.4"
BUNDLE_ID="com.spacesoftware.factux"
MAIN_CLASS="electricpos.Main"
JAR_FILE="ElectricPos.jar"
ICON_FILE="app.icns"
MIN_OSX_VERSION="10.9"

# Rutas (ajusta según tu entorno)
JRE_PATH="/Library/Java/JavaVirtualMachines/temurin-8.jdk/Contents/Home"

# ============================================================================
# NO MODIFICAR DEBAJO DE ESTA LÍNEA (A MENOS QUE SEPAS LO QUE HACES)
# ============================================================================

# Colores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Función para imprimir mensajes
print_step() {
    echo -e "${BLUE}[$1]${NC} $2"
}

print_success() {
    echo -e "${GREEN}✓${NC} $1"
}

print_error() {
    echo -e "${RED}✗ ERROR:${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠ ADVERTENCIA:${NC} $1"
}

# ============================================================================
# INICIO
# ============================================================================

echo ""
echo "============================================"
echo "  Crear Instalador macOS"
echo "  ${APP_NAME} v${APP_VERSION}"
echo "============================================"
echo ""

# ============================================================================
# PASO 1: VERIFICACIONES
# ============================================================================

print_step "1/9" "Verificando requisitos..."

# Verificar que estamos en Mac
if [[ "$OSTYPE" != "darwin"* ]]; then
    print_error "Este script debe ejecutarse en macOS"
    exit 1
fi

# Verificar JAR
if [ ! -f "${JAR_FILE}" ]; then
    print_error "No se encuentra ${JAR_FILE}"
    echo "Por favor, compila el proyecto primero y asegúrate de estar en el directorio correcto"
    exit 1
fi
print_success "JAR encontrado: ${JAR_FILE}"

# Verificar librerías
if [ ! -d "lib" ]; then
    print_error "No se encuentra la carpeta 'lib' con las librerías"
    exit 1
fi
print_success "Librerías encontradas: $(ls lib/*.jar 2>/dev/null | wc -l | tr -d ' ') archivos"

# Verificar JRE
if [ ! -d "${JRE_PATH}" ]; then
    print_error "No se encuentra Java 8 JRE en ${JRE_PATH}"
    echo ""
    echo "Para instalar Java 8 ejecuta:"
    echo "  brew tap homebrew/cask-versions"
    echo "  brew install --cask temurin8"
    echo ""
    exit 1
fi
print_success "JRE encontrado: ${JRE_PATH}"

# Verificar icono (opcional)
if [ ! -f "${ICON_FILE}" ]; then
    print_warning "No se encuentra ${ICON_FILE}, se usará icono por defecto"
    ICON_FILE=""
else
    print_success "Icono encontrado: ${ICON_FILE}"
fi

echo ""

# ============================================================================
# PASO 2: CREAR ESTRUCTURA
# ============================================================================

print_step "2/9" "Creando estructura de carpetas..."

BUNDLE_DIR="${APP_NAME}.app"

# Eliminar bundle anterior si existe
if [ -d "${BUNDLE_DIR}" ]; then
    print_warning "Eliminando bundle anterior..."
    rm -rf "${BUNDLE_DIR}"
fi

# Crear estructura
mkdir -p "${BUNDLE_DIR}/Contents/MacOS"
mkdir -p "${BUNDLE_DIR}/Contents/Resources"
mkdir -p "${BUNDLE_DIR}/Contents/PlugIns"

print_success "Estructura creada"

# ============================================================================
# PASO 3: COPIAR JAR Y LIBRERÍAS
# ============================================================================

print_step "3/9" "Copiando JAR y librerías..."

cp "${JAR_FILE}" "${BUNDLE_DIR}/Contents/Resources/"
cp -r lib "${BUNDLE_DIR}/Contents/Resources/"

print_success "Archivos copiados"

# ============================================================================
# PASO 4: COPIAR ICONO
# ============================================================================

print_step "4/9" "Copiando icono..."

if [ -n "${ICON_FILE}" ]; then
    cp "${ICON_FILE}" "${BUNDLE_DIR}/Contents/Resources/"
    print_success "Icono copiado"
else
    print_warning "Sin icono personalizado"
fi

# ============================================================================
# PASO 5: EMBEBED JRE (puede tardar)
# ============================================================================

print_step "5/9" "Embebiendo JRE (esto puede tomar varios minutos)..."

cp -r "${JRE_PATH}" "${BUNDLE_DIR}/Contents/PlugIns/jre.jdk"

# Calcular tamaño del JRE
JRE_SIZE=$(du -sh "${BUNDLE_DIR}/Contents/PlugIns/jre.jdk" | cut -f1)
print_success "JRE embebido (tamaño: ${JRE_SIZE})"

# ============================================================================
# PASO 6: CREAR LAUNCHER SCRIPT
# ============================================================================

print_step "6/9" "Creando launcher script..."

cat > "${BUNDLE_DIR}/Contents/MacOS/FactuxLauncher" << 'LAUNCHER_EOF'
#!/bin/bash

# Obtener directorio de la aplicación
DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
APP_ROOT="$(dirname "$DIR")"

# Configurar Java
JAVA_HOME="${APP_ROOT}/PlugIns/jre.jdk"
JAVA="${JAVA_HOME}/bin/java"

# Verificar que Java existe
if [ ! -x "${JAVA}" ]; then
    osascript -e 'display dialog "Error: No se encuentra Java en la aplicación.\n\nPor favor, reinstala Factux Costa Rica." buttons {"OK"} default button 1 with icon stop'
    exit 1
fi

# Construir classpath
CLASSPATH="${APP_ROOT}/Resources/ElectricPos.jar"
for jar in "${APP_ROOT}/Resources/lib"/*.jar; do
    if [ -f "$jar" ]; then
        CLASSPATH="${CLASSPATH}:${jar}"
    fi
done

# Opciones de JVM
JAVA_OPTS="-Xms256m -Xmx1024m"
JAVA_OPTS="${JAVA_OPTS} -Dfile.encoding=UTF-8"
JAVA_OPTS="${JAVA_OPTS} -Duser.language=es"
JAVA_OPTS="${JAVA_OPTS} -Duser.country=CR"

# Opciones específicas de macOS
JAVA_OPTS="${JAVA_OPTS} -Dapple.laf.useScreenMenuBar=true"
JAVA_OPTS="${JAVA_OPTS} -Dcom.apple.mrj.application.apple.menu.about.name=Factux Costa Rica"
JAVA_OPTS="${JAVA_OPTS} -Dapple.awt.application.name=Factux Costa Rica"

# Ejecutar aplicación
"${JAVA}" ${JAVA_OPTS} -cp "${CLASSPATH}" electricpos.Main

# Capturar código de salida
EXIT_CODE=$?

# Si hubo error, mostrar diálogo
if [ ${EXIT_CODE} -ne 0 ]; then
    osascript -e "display dialog \"La aplicación se cerró inesperadamente.\n\nCódigo de error: ${EXIT_CODE}\" buttons {\"OK\"} default button 1 with icon caution"
fi

exit ${EXIT_CODE}
LAUNCHER_EOF

# Dar permisos de ejecución
chmod +x "${BUNDLE_DIR}/Contents/MacOS/FactuxLauncher"

print_success "Launcher creado y configurado"

# ============================================================================
# PASO 7: CREAR INFO.PLIST
# ============================================================================

print_step "7/9" "Creando Info.plist..."

ICON_KEY=""
if [ -n "${ICON_FILE}" ]; then
    ICON_KEY="<key>CFBundleIconFile</key>
    <string>${ICON_FILE}</string>"
fi

cat > "${BUNDLE_DIR}/Contents/Info.plist" << PLIST_EOF
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>CFBundleName</key>
    <string>${APP_NAME}</string>
    <key>CFBundleDisplayName</key>
    <string>${APP_NAME}</string>
    <key>CFBundleIdentifier</key>
    <string>${BUNDLE_ID}</string>
    <key>CFBundleVersion</key>
    <string>${APP_VERSION}</string>
    <key>CFBundleShortVersionString</key>
    <string>${APP_VERSION}</string>
    <key>CFBundlePackageType</key>
    <string>APPL</string>
    <key>CFBundleSignature</key>
    <string>FACT</string>
    <key>CFBundleExecutable</key>
    <string>FactuxLauncher</string>
    ${ICON_KEY}
    <key>NSHighResolutionCapable</key>
    <true/>
    <key>NSSupportsAutomaticGraphicsSwitching</key>
    <true/>
    <key>LSMinimumSystemVersion</key>
    <string>${MIN_OSX_VERSION}</string>
    <key>NSHumanReadableCopyright</key>
    <string>© 2025 Space Software Costa Rica</string>
</dict>
</plist>
PLIST_EOF

print_success "Info.plist creado"

# ============================================================================
# PASO 8: VERIFICAR APP
# ============================================================================

print_step "8/9" "Verificando aplicación..."

# Verificar permisos
if [ -x "${BUNDLE_DIR}/Contents/MacOS/FactuxLauncher" ]; then
    print_success "Launcher es ejecutable"
else
    print_error "Launcher no tiene permisos de ejecución"
    exit 1
fi

# Verificar estructura
REQUIRED_FILES=(
    "${BUNDLE_DIR}/Contents/Info.plist"
    "${BUNDLE_DIR}/Contents/MacOS/FactuxLauncher"
    "${BUNDLE_DIR}/Contents/Resources/${JAR_FILE}"
    "${BUNDLE_DIR}/Contents/PlugIns/jre.jdk/bin/java"
)

for file in "${REQUIRED_FILES[@]}"; do
    if [ ! -e "$file" ]; then
        print_error "Falta archivo requerido: $file"
        exit 1
    fi
done

print_success "Estructura verificada"

# ============================================================================
# PASO 9: CREAR DMG
# ============================================================================

print_step "9/9" "Creando DMG..."

DMG_NAME="FactuxCostaRica-${APP_VERSION}.dmg"
DMG_TEMP="dmg_temp"

# Limpiar archivos anteriores
rm -f "${DMG_NAME}"
rm -rf "${DMG_TEMP}"

# Crear carpeta temporal
mkdir -p "${DMG_TEMP}"

# Copiar aplicación
cp -r "${BUNDLE_DIR}" "${DMG_TEMP}/"

# Crear enlace simbólico a Applications
ln -s /Applications "${DMG_TEMP}/Applications"

# Crear DMG
hdiutil create -volname "${APP_NAME} ${APP_VERSION}" \
  -srcfolder "${DMG_TEMP}" \
  -ov -format UDZO \
  "${DMG_NAME}" > /dev/null 2>&1

# Limpiar
rm -rf "${DMG_TEMP}"

# Calcular tamaño del DMG
DMG_SIZE=$(du -sh "${DMG_NAME}" | cut -f1)
print_success "DMG creado (tamaño: ${DMG_SIZE})"

# ============================================================================
# RESUMEN FINAL
# ============================================================================

echo ""
echo "============================================"
echo "  ✅ INSTALADOR CREADO EXITOSAMENTE"
echo "============================================"
echo ""
echo "Archivos generados:"
echo "  📁 ${BUNDLE_DIR}"
echo "  💿 ${DMG_NAME}"
echo ""
echo "Información:"
echo "  Versión:        ${APP_VERSION}"
echo "  Tamaño DMG:     ${DMG_SIZE}"
echo "  Bundle ID:      ${BUNDLE_ID}"
echo "  macOS mínimo:   ${MIN_OSX_VERSION}"
echo ""
echo "Para probar la aplicación:"
echo "  open '${BUNDLE_DIR}'"
echo ""
echo "Para distribuir:"
echo "  Usa el archivo: ${DMG_NAME}"
echo ""

# Verificar firma
if codesign -v "${BUNDLE_DIR}" 2>/dev/null; then
    print_success "Aplicación firmada"
else
    print_warning "Aplicación NO firmada"
    echo ""
    echo "ℹ️  NOTA IMPORTANTE:"
    echo "    Para distribución pública en macOS 10.15+, debes:"
    echo "    1. Firmar la aplicación con un certificado de desarrollador"
    echo "    2. Notarizar la aplicación con Apple"
    echo ""
    echo "    Sin esto, los usuarios verán advertencias de seguridad."
    echo "    Ver documentación: GUIA_CREAR_INSTALADOR_MAC.md"
fi

echo ""
echo "¿Deseas abrir el DMG ahora? (s/n)"
read -r response
if [[ "$response" =~ ^([sS][iI]|[sS])$ ]]; then
    open "${DMG_NAME}"
fi

echo ""
echo "============================================"
echo "  Factux Costa Rica © 2025"
echo "  Space Software Costa Rica"
echo "============================================"
echo ""

exit 0


